package org.nhindirect.monitor.processor.impl;

import java.util.Arrays;
import java.util.Collection;
import java.util.Map;
import java.util.Set;

import javax.mail.internet.MimeMessage;

import org.apache.camel.Exchange;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nhindirect.common.mail.MDNStandard;
import org.nhindirect.common.tx.TxUtil;
import org.nhindirect.common.tx.impl.DefaultTxDetailParser;
import org.nhindirect.common.tx.model.Tx;
import org.nhindirect.common.tx.model.TxDetail;
import org.nhindirect.common.tx.model.TxDetailType;
import org.nhindirect.common.tx.model.TxMessageType;
import org.nhindirect.monitor.condition.impl.AbstractCompletionCondition;
import org.nhindirect.monitor.dao.NotificationDAOException;
import org.nhindirect.monitor.processor.DuplicateNotificationStateManagerException;

/**
 * Extended implementation of the DefaultDuplicateNotificationStateManager that handles DSN messages generated by 
 * the message monitor and properly handles tracking of DSN messages generated by the message monitor.
 * @author Greg Meyer
 * @since 1.1.4
 */
public class TimeoutDupStateManager extends DefaultDuplicateNotificationStateManager
{
	private static final Log LOGGER = LogFactory.getFactory().getInstance(TimeoutDupStateManager.class);
	
	protected final DefaultTxDetailParser parser;
	
	/**
	 * Default contructor
	 */
	public TimeoutDupStateManager()
	{
		super();
		
		parser = new DefaultTxDetailParser();
	}
	
	
	/**
	 * Adds a duplication state entry for each recipient of the original message
	 * @param txs Collection of message transactions.  The original message must be present in the collection
	 * @throws DuplicateNotificationStateManagerException
	 */
	public void addNotificationForOriginalRecips(Collection<Tx> txs) throws DuplicateNotificationStateManagerException
	{
		if (dao == null)
			throw new IllegalArgumentException("Dao cannot be null");	
		
		if (txs == null)
			throw new IllegalArgumentException("Txs cannot be null");
		
		final Tx tx = AbstractCompletionCondition.getMessageToTrack(txs);
		
		if (tx != null && TxUtil.isReliableAndTimelyRequested(tx))
		{
			final TxDetail recipDetail = tx.getDetail(TxDetailType.RECIPIENTS);
			final TxDetail msgId = tx.getDetail(TxDetailType.MSG_ID);
			
			if (recipDetail != null && msgId != null)
			{
				final String[] recips = recipDetail.getDetailValue().split(",");
				for (String recip : recips)
				{
					
					try
					{
						final String normalizedFinalRecip = AbstractCompletionCondition.normalizeFinalRecip(recip);
						dao.addNotification(msgId.getDetailValue(), normalizedFinalRecip);
					}
					///CLOVER:OFF
					catch (NotificationDAOException e)
					{
						LOGGER.warn("Can't add recipient " + recip + " for message id " + msgId.getDetailValue() 
								+ " to dup store.  Possibly already exists in dup store.");
					}
					///CLOVER:ON
				}
			}
		}
	}
	
	/**
	 * Adds a duplication state entry for each final recipient in a DSN message that was generated by the message monitor
	 * @param dsnMessage The DSN message
	 * @param ex The aggregation exchange.  The property "ORGI_IS_TIMELY" is checked to see if this DSN is part of a timely and reliable
	 * exchange
	 * @throws DuplicateNotificationStateManagerException
	 */
	public void addNotificationForMonitorGeneratedDSN(MimeMessage dsnMessage, Exchange ex) throws DuplicateNotificationStateManagerException
	{
		if (dao == null)
			throw new IllegalArgumentException("Dao cannot be null");	
		
		if (dsnMessage == null)
			throw new IllegalArgumentException("DSN message cannot be null");
		
		if (TxUtil.getMessageType(dsnMessage).equals(TxMessageType.DSN) && ex.getProperty("ORIG_IS_TIMELY") != null)
		{
			final Map<String, TxDetail> details = parser.getMessageDetails(dsnMessage);
			
			final TxDetail origMsgId = details.get(TxDetailType.PARENT_MSG_ID.getType());
			final TxDetail finalRecipDetail = details.get(TxDetailType.FINAL_RECIPIENTS.getType());
			final TxDetail msgId = details.get(TxDetailType.MSG_ID.getType());
			
			if (origMsgId != null && finalRecipDetail != null && msgId != null)
			{
				final String[] finalRecips = finalRecipDetail.getDetailValue().split(",");
				for (String finalRecip : finalRecips)
				{
					try
					{
						final String normalizedFinalRecip = AbstractCompletionCondition.normalizeFinalRecip(finalRecip);
						final String addMsgId = origMsgId.getDetailValue() + "\t" + msgId.getDetailValue();
						dao.addNotification(addMsgId, normalizedFinalRecip);
						
						// also write it out with just the orig message id
						// this is needed for DSN messages that are not generated by the message monitor
						// so we can ensure that later messages are properly suppressed
						dao.addNotification(origMsgId.getDetailValue(), normalizedFinalRecip);
					}
					///CLOVER:OFF
					catch (NotificationDAOException e)
					{
						LOGGER.warn("Can't add recipient " + finalRecip + " for message id " + msgId.getDetailValue() 
								+ " to dup store.  Possibly already exists in dup store.");
					}
					///CLOVER:ON
				}
			}
		}
		
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean suppressNotification(Tx notificationMessage) throws DuplicateNotificationStateManagerException
	{
		boolean retVal = false;
		
		if (dao == null)
			throw new IllegalArgumentException("Dao cannot be null");	
		
		if (notificationMessage == null)
			throw new IllegalArgumentException("Notification message cannot be null");
		
		final TxMessageType type = notificationMessage.getMsgType();

		
		if (type == TxMessageType.DSN || type == TxMessageType.MDN)
		{
			final TxDetail dispositionDetail = notificationMessage.getDetail(TxDetailType.DISPOSITION);
			
			// if it's an MDN displayed, then don't suppress it and let it go through
			if (type == TxMessageType.MDN && (dispositionDetail == null || dispositionDetail.getDetailValue().contains(MDNStandard.Disposition_Displayed)))
				return retVal;
			
			final TxDetail originalMessageIdDetail = notificationMessage.getDetail(TxDetailType.PARENT_MSG_ID);
			final TxDetail origRecips = notificationMessage.getDetail(TxDetailType.FINAL_RECIPIENTS);
			
			if (originalMessageIdDetail != null && origRecips != null)
			{
				final String[] recipsArray = origRecips.getDetailValue().split(",");
				for (int idx = 0; idx < recipsArray.length; ++ idx)
					recipsArray[idx] = AbstractCompletionCondition.normalizeFinalRecip(recipsArray[idx]);
				
				final Collection<String> recips = Arrays.asList(recipsArray);
				
				try
				{	
					// try by tabbular concatenation first... we need to make sure DSN messages
					// generated by the message monitor are not supressed... these messages
					// should be the only ones that use the tabbular format
					final String queryMsgId = originalMessageIdDetail.getDetailValue() + "\t" + notificationMessage.getDetail(TxDetailType.MSG_ID).getDetailValue();
					
					Set<String> alreadyReceivedNotifs = dao.getReceivedAddresses(queryMsgId, recips);
					if (!alreadyReceivedNotifs.isEmpty())
						// this is from a DSN message generated by the monitor... don't suppress it
						retVal = false;
					else
					{
						// ok, so it's not in the tabular list... so just do the normal suppression test
						alreadyReceivedNotifs = dao.getReceivedAddresses(originalMessageIdDetail.getDetailValue(), recips);
						if (!alreadyReceivedNotifs.isEmpty())
							retVal = true;
					}
	
					
				}
				catch (NotificationDAOException e)
				{
					throw new DuplicateNotificationStateManagerException(e);
				}
				
			}
		}
		
		return retVal;
	}	
}
